// Expected global variables:
/*global clipboardData domExplorerCode jQuery */

(function ($) {
    /// <summary>
    ///     TabItem - JQuery extension for constructing a tab item, 
    ///     The tabItem object can be initialized by calling tabItem on a jQuery object.
    ///     Requires:
    ///     - layout.css
    /// </summary>
    
    var methods = {
    
        // Default constructor
        init: function (selectedCallback) {
            /// <summary>
            ///     Creates a new TabItem
            /// </summary>
            /// <param name="selectedCallback" type="Function">
            ///     The callback to fire when the tab is changed
            /// </param>
            /// <returns type="Object">
            ///     The jquery object for the new TabItem
            /// </returns>
            
            this.addClass("BPT-Tab-Item");
            
            // Add the click and focus handlers
            this.bind("click focus", function () {
            
                // Ensure we aren't currently editing a textbox
                if (document.activeElement && document.activeElement.type !== "text") {
                    var item = $(this);
                    
                    // Get the currently selected tab from the siblings of the one clicked (or focused)
                    var currentElement = item.siblings(".BPT-Tab-Item.BPT-Tab-Item-Active");

                    // Ensure the tab has changed
                    if (currentElement[0]) {
                        currentElement.removeClass("BPT-Tab-Item-Active");
                        currentElement.addClass("BPT-Tab-Item-Inactive");
                        currentElement.removeAttr("tabindex");
                        
                        item.attr("tabindex", "1");
                        item.removeClass("BPT-Tab-Item-Inactive");
                        item.addClass("BPT-Tab-Item-Active");
                        if (selectedCallback) {
                            selectedCallback();
                        }
                    }
                }
            });
        },
        
        isActive: function () {
            /// <summary>
            ///     Gets a value indicating if this TabItem is the currently selected one
            /// </summary>  
            /// <returns type="Boolean">
            ///     True if this item is active, false otherwise
            /// </returns>
            
            return this.hasClass("BPT-Tab-Item-Active");
        }
    };

    $.fn.tabItem = function (method) {
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof method !== "object" || !method) {
            return methods.init.apply(this, arguments);
        }
    };
}(jQuery));


(function ($) {
    /// <summary>
    ///     HorizontalPane - JQuery extension for constructing a horizontal resizable pane, 
    ///     The horizontalPane object can be initialized by calling horizontalPane on a jQuery object.
    ///     Requires:
    ///     - layout.css
    /// </summary>

    var methods = {
    
        // Default constructor
        init: function () {
            /// <summary>
            ///     Creates a new HorizontalPane
            /// </summary>
            /// <returns type="Object">
            ///     The jquery object for the new HorizontalPane
            /// </returns>
            
            var item = this;
            var rightPane = item.children(".BPT-HorizontalPane-Right");
            item.children().addClass("BPT-Pane");

            var windowWidth = $(window).width();
            
            var curWidth = (windowWidth > 0 ? (windowWidth / 100) * 30 : 300);
            var minPaneSize = 200;
            var setPaneWidth = function (newWidth) {
                if (newWidth <= 0) {
                    return;
                }
                curWidth = newWidth = Math.max(minPaneSize, newWidth);
                curWidth = newWidth = Math.round(curWidth); // Clamp to nearest pixel
                var paneSize = item.outerWidth();
                if (paneSize < minPaneSize * 2) {
                    newWidth = paneSize >> 1;
                } else if (paneSize - newWidth < minPaneSize) {
                    newWidth = paneSize - minPaneSize;
                }
                
                item.css("paddingRight", newWidth + "px");
                rightPane.css("marginRight", "-" + newWidth + "px");
                rightPane.css("width", newWidth + "px");
            };
            
            $(window).resize(function () {
                setPaneWidth(curWidth);
            });
            
            setPaneWidth(curWidth);

            var divider = $("<div class='BPT-Pane-Divider'></div>");
            rightPane.prepend(divider);
            divider.mousedown(function (evt) {
                var prevCursor = document.body.style.cursor;
                document.body.style.cursor = "w-resize";

                // Create mouse handlers for resizing
                var mouseMoveHandler, mouseUpHandler;
                mouseMoveHandler = function (evt) {
                    // If the user triggered the 'mouseup' event outside the tool window
                    if (!window.event.button) {
                        mouseUpHandler();
                    }
                    var dividerLoc = divider.offset().left;
                    setPaneWidth(rightPane.width() - evt.pageX + dividerLoc);
                };
                mouseUpHandler = function () {
                    $(document).unbind("mousemove", mouseMoveHandler);
                    $(document).unbind("mouseup", mouseUpHandler);
                    document.body.style.cursor = prevCursor;
                };
                $(document).bind("mousemove", mouseMoveHandler);
                $(document).bind("mouseup", mouseUpHandler);
                
                // Prevent highlighting text while resizing
                // This also stops resizing while the cursor is outside our window.
                evt.stopImmediatePropagation();
                evt.preventDefault();
            });

        }
    };
    
    $.fn.horizontalPane = function (method) {
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof method !== "object" || !method) {
            return methods.init.apply(this, arguments);
        }
    };    
        
}(jQuery));


(function ($) {
    /// <summary>
    ///     EditTextbox - JQuery extension for constructing a double click editbox,
    ///     Takes a span element and replaces it with a textbox that has the same contents as the original span.  
    ///     Triggers a callback function when the changes are being submitted.    
    ///     The EditTextbox object can be initialized by calling editTextbox on a jQuery object.
    ///
    ///     Requires:
    ///     - layout.css
    /// </summary>
    
    var textBox = $("<input type='text' class='BPT-EditBox'></input>");
    var replacedObj = null;
    var editChangeCallback = null;
    var editClosedCallback = null;
    var originalValue = null;
    var originalOverflow = "";
    var lastCommitedValue = null;
    var arrowKeysUsed = false;
    var elementRemoved = false;
    
    var methods = {
    
        // Default constructor
        init: function (changeCallback, closedCallback, dataAttributes) {
            /// <summary>
            ///     Creates a new EditTextbox
            /// </summary>
            /// <param name="changeCallback" type="Function" optional="true">
            ///     Optional parameter specifying a callback for when the edit box has changed text
            /// </param> 
            /// <param name="closedCallback" type="Function" optional="true">
            ///     Optional parameter specifying a callback for when the edit box has closed
            /// </param>     
            /// <param name="dataAttributes" type="Array" optional="true">
            ///     Optional parameter specifying an array of data-attributes to be applied to the textbox for identification
            /// </param>             
            /// <returns type="Object">
            ///     The jquery object for the new EditTextbox
            /// </returns>
            
            var removeAndFocusParentContainer, removeTextbox;
            removeAndFocusParentContainer = function (evt, wasCancelled) {
                var container = textBox.parents(".BPT-DataTree-Container");
                container = (container.length > 0 ? container : textBox.parents(".BPT-HtmlTree-Container"));
                container = (container.length > 0 ? container : textBox.parents(".BPT-EditContainer:first"));
                
                removeTextbox(wasCancelled);

                if (container.length > 0) {
                    // We need to use 'setTimeout' to ensure that the textbox has been removed before changing the active element back to the container
                    window.setTimeout(function () {
                        // Ensure that the focus is not already on another textbox before moving focus to the container
                        if (document.activeElement && document.activeElement.type !== "text") {
                            try {
                                container.data("mouseActivate", true); // Stop the default 'scroll into view' behavior
                                container[0].setActive();
                            } catch (ex) {
                                // Setting the active element can sometimes cause an 'Incorrect Function' exception in IE9/10,
                                // We should fail gracefully in this situation.
                            }
                        }
                    }, 0);
                }
            };
            
            removeTextbox = function (wasCancelled) {
                $(document).unbind("mousedown", removeAndFocusParentContainer);
                
                var newValue = textBox.attr("value");
                
                if (wasCancelled) {
                    // We need to reset the value because it was cancelled
                    newValue = originalValue;
                    
                    // We need to fire another change, if the arrow keys were used to change the value dynamically
                    if (arrowKeysUsed && lastCommitedValue !== originalValue) {
                        if (editChangeCallback) {
                            editChangeCallback(originalValue);
                        }
                    }
                } else if (arrowKeysUsed && lastCommitedValue !== newValue) {
                    // Fire a change because the arrow keys changed the value dynamically, but then the user changed the text manually
                    if (editChangeCallback) {
                        editChangeCallback(newValue);
                    }
                } else if (newValue !== originalValue) {
                    // Fire a change because the user changed the text manually
                    elementRemoved = false;
                    if (editChangeCallback) {
                        editChangeCallback(newValue);
                    }
                } else if (newValue === originalValue) {
                    // If this wasn't canceled then we shouldn't remove the element
                    elementRemoved = false;
                }
                
                // Update the real value
                if (replacedObj) {
                    // Restore the parent's style
                    textBox.parent().css("overflow", originalOverflow);
            
                    textBox.replaceWith(replacedObj);
                    replacedObj.text(newValue);
                    replacedObj = null;

                    // Fire the closed callback
                    if (editClosedCallback) {
                        editClosedCallback(newValue, wasCancelled, elementRemoved);
                    }
                }
            };
           
            var stopPropagationHandler = function (evt) {
                evt.stopPropagation();
            };
            
            var keyPressHandler = function (evt) {
                if (evt.keyCode === 9 || evt.keyCode === 13 || evt.keyCode === 27) { // Tab(9), Enter(13), Escape(27)
                
                    removeAndFocusParentContainer(evt, evt.keyCode === 27);   // Escape(27) means the commit was cancelled
                    evt.stopImmediatePropagation();
                    return false;
                } else if (event.keyCode === 38 || event.keyCode === 40) { // Up(38) or Down(40)
                
                    // Get the value and check for a single number
                    var text = textBox.val();
                    var number = parseInt(text, 10);
                    if (!isNaN(number)) {

                        // Change the number
                        var changeAmount = (evt.shiftKey ? 10 : 1);
                        var newNumber = number + (event.keyCode === 38 ? (1 * changeAmount) : (-1 * changeAmount));
                        var newText = text.replace("" + number, newNumber);
                        textBox.val(newText);
                        
                        // Fire the callback
                        if (editChangeCallback) {
                            // Mark the usage of arrow keys to change the value
                            arrowKeysUsed = true;
                            lastCommitedValue = newText;
                            editChangeCallback(newText);
                        }
                    }            
                }
            };
            
            var valueChangedHandler = function (e, newValue) {
                // The original value has been changed (usually due to a mutation event occurring while the item is being edited)
                originalValue = newValue;
                elementRemoved = false;
            };

            var valueRemovedHandler = function (e) {
                // The original element has been removed while we are editing it, so flag that we need to remove it after the edit completes
                elementRemoved = true;
            };
            
            // If we are trying to create an edit box for a non-existent element, then quit early
            if (this.length === 0) {
                return;
            }
            
            // Remove any existing textbox
            if (replacedObj) {
                // Inform caller that the commit was cancelled
                removeTextbox(true);
            }

            // Set the variables and create the textbox
            editChangeCallback = changeCallback;
            editClosedCallback = closedCallback;
            originalValue = this.text();
            lastCommitedValue = null;
            arrowKeysUsed = false;
            elementRemoved = false;
            
            // Try to have the textbox at around the same size as the thing we are replacing
            textBox.attr("size", Math.max(originalValue.length, 8));
            
            // Save the parent's original style and change it to always allow the textbox to be seen
            originalOverflow = this.parent().css("overflow");
            this.parent().css("overflow", "visible");

            // Remove the previous data attributes
            var i = 0;
            for (i = 0; i < textBox[0].attributes.length; i++) {
                var name = textBox[0].attributes[i].name;
                if (name.indexOf("data-") === 0) {
                    textBox.removeAttr(name);
                }
            }
            // Add any new ones
            if (dataAttributes) {
                for (i = 0; i < dataAttributes.length; i++) {
                    var attribute = dataAttributes[i];
                    textBox.attr(attribute.name, attribute.value);
                }
            }
            
            replacedObj = this.replaceWith(textBox.attr("value", originalValue));
            
            // Add event handlers
            $(document).bind("mousedown", removeAndFocusParentContainer, true);
            textBox.bind("keydown", keyPressHandler);
            textBox.bind("mousedown mouseup click dblclick", stopPropagationHandler);
            textBox.bind("valueChanged", valueChangedHandler);
            textBox.bind("valueRemoved", valueRemovedHandler);
            
            textBox.bind("contextmenu", function (e) {
                var x = e.clientX;
                var y = e.clientY;
                if (e.clientX <= 0 || e.clientY <= 0) {
                    // Keyboard activation
                    x = textBox.offset().left;
                    y = textBox.offset().top;
                }
                
                // Get the context menu parameters
                var selectedText = document.selection.createRange().text;
                var pasteText = clipboardData.getData("Text");
                var canCut = (selectedText !== "");
                var canCopy = canCut;
                var canPaste = (pasteText && pasteText !== "" ? true : false);
                var menuParams = [
                    canCut,
                    canCopy,
                    canPaste
                ];
                
                var callback = function (id, selectedMenuItem) {
                    if (id === "menuTextControl") {
                        switch (selectedMenuItem) {
                            case "menuTextControlCut":
                                document.selection.createRange().execCommand("Cut");
                                break;
                                
                            case "menuTextControlCopy":
                                clipboardData.setData("Text", selectedText);
                                break;
                                
                            case "menuTextControlPaste":
                                var textElement = textBox[0];
                                var startPos = textElement.selectionStart;
                                var endPos = textElement.selectionEnd;

                                textElement.value = textElement.value.substring(0, startPos) + pasteText + textElement.value.substring(endPos, textElement.value.length);
                                break;
                        }
                    }
                };
                domExplorerCode.externalApis.vsBridge.showContextMenu("menuTextControl", x, y, callback, menuParams);

                // Stop the real context menu
                stopPropagationHandler(e);
                return false;
            });

            // Now select and focus the new textbox
            textBox.select().focus();
        }
    };
    
    $.fn.editTextbox = function (method) {
        if (methods[method]) {
            return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
        } else if (typeof method !== "object" || !method) {
            return methods.init.apply(this, arguments);
        }
    };      
}(jQuery));

// SIG // Begin signature block
// SIG // MIIaiQYJKoZIhvcNAQcCoIIaejCCGnYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFMsXdEIvxMZa
// SIG // q9IToK94qdae/VvloIIVeTCCBLowggOioAMCAQICCmEC
// SIG // kkoAAAAAACAwDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OVoXDTEz
// SIG // MDQwOTIyMjU1OVowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpCOEVDLTMwQTQtNzE0NDElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAM1jw/ei
// SIG // tUfZ+TmUU6xrj6Z5OCH00W49FTgWwXMsmY/74Dxb4aJM
// SIG // i7Kri7TySse5k1DRJvWHU7B6dfNHDxcrZyxk62DnSozg
// SIG // i17EVmk3OioEXRcByL+pt9PJq6ORqIHjPy232OTEeAB5
// SIG // Oc/9x2TiIxJ4ngx2J0mPmqwOdOMGVVVJyO2hfHBFYX6y
// SIG // cRYe4cFBudLSMulSJPM2UATX3W88SdUL1HZA/GVlE36V
// SIG // UTrV/7iap1drSxXlN1gf3AANxa7q34FH+fBSrubPWqzg
// SIG // FEqmcZSA+v2wIzBg6YNgrA4kHv8R8uelVWKV7p9/ninW
// SIG // zUsKdoPwQwTfBkkg8lNaRLBRejkCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBTNGaxhTZRnK/avlHVZ2/BYAIOhOjAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBRHNbfNh3cgLwCp8aZ3xbI
// SIG // kAZpFZoyufNkENKK82IpG3mPymCps13E5BYtNYxEm/H0
// SIG // XGGkQa6ai7pQ0Wp5arNijJ1NUVALqY7Uv6IQwEfVTnVS
// SIG // iR4/lmqPLkAUBnLuP3BZkl2F7YOZ+oKEnuQDASETqyfW
// SIG // zHFJ5dod/288CU7VjWboDMl/7jEUAjdfe2nsiT5FfyVE
// SIG // 5x8a1sUaw0rk4fGEmOdP+amYpxhG7IRs7KkDCv18elId
// SIG // nGukqA+YkqSSeFwreON9ssfZtnB931tzU7+q1GZQS/DJ
// SIG // O5WF5cFKZZ0lWFC7IFSReTobB1xqVyivMcef58Md7kf9
// SIG // J9d/z3TcZcU/MIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggR8MIIEeAIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGeMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTOqS5j3hkSLVWeuWKu/Et2
// SIG // t85sYzA+BgorBgEEAYI3AgEMMTAwLqAUgBIAbABhAHkA
// SIG // bwB1AHQALgBqAHOhFoAUaHR0cDovL21pY3Jvc29mdC5j
// SIG // b20wDQYJKoZIhvcNAQEBBQAEggEAuDat/4p23KBpA9t0
// SIG // DU8FEAGLFHq7FCXKSv3Hlo8JGDE4twaHl6E7tnzayZwX
// SIG // nssPBL7gByX9SOZHuZKCtZ1EILrnOuI4LRpDXun6V3k1
// SIG // fQrAH+3KB3tjZTRvtxX50vrCiMHbQganDmnvJFMiw+nw
// SIG // QOZjs/WsWLvVnAXcCPckzVjNCY9KtA7/SHKk4c3G1IkF
// SIG // FQULiPjv61TD4XA11A/wrxYii8vhPTEB9K55I08WrfmW
// SIG // CdPhS5TmINqeMMU9qUDAoYkDyXMvcYsJi8KzvBuRr3/U
// SIG // R9qWn4stZgbPyUTO80/Opa18F1m3Roi+jiJKwFOWdxGg
// SIG // dbKcAJQQc+n+5J5ldqGCAh8wggIbBgkqhkiG9w0BCQYx
// SIG // ggIMMIICCAIBATCBhTB3MQswCQYDVQQGEwJVUzETMBEG
// SIG // A1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
// SIG // ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
// SIG // MSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQ
// SIG // Q0ECCmECkkoAAAAAACAwCQYFKw4DAhoFAKBdMBgGCSqG
// SIG // SIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkF
// SIG // MQ8XDTEzMDMxNTA2MzQwNFowIwYJKoZIhvcNAQkEMRYE
// SIG // FNmkQnfuabB6qe7wQWHZN9uN7eOrMA0GCSqGSIb3DQEB
// SIG // BQUABIIBADPn4Bxtl01YDdIfrpQ9w//U2GtXp7p9IgHn
// SIG // a++nScR89MoOEkMNCR08+Yep9vKu1ywlFYbhrgV53Oiv
// SIG // xng2NyNuAcGbDCnLuj3MyW6MVGLqMVPg2vxWogS/VR3W
// SIG // MQuvH6bxhFJ8zy0V2bmQq5y2E0I46ssuWxautUD1bF/w
// SIG // fuUysAgxMAMHOQXYyhTvRdwq4/SGmWLxOSyOgKWmkM89
// SIG // e/rWw4A/CPhWMEbmufCOT07nmXrmGRNlkBkOv4C8rx+O
// SIG // yuf00kn25ZKN4hLNsnQZtZtGHy3M3c/HdG6wPVN+ecNP
// SIG // Fu/u8eiA20Kp09Ct63X6VlCXwTUHZxDX6gETuN4PlIM=
// SIG // End signature block
